package com.ruoyi.grpc.service;

import com.google.gson.Gson;
import com.google.protobuf.ByteString;
import com.ruoyi.file.enums.MsgFileType;
import com.ruoyi.file.enums.MsgHandlerType;
import com.ruoyi.grpc.file.*;
import com.ruoyi.user.dto.ClientUserOnline;
import com.ruoyi.user.service.TokenApiService;
import com.ruoyi.watch.service.INmyUserWatchSubscribeService;
import com.ruoyi.websocket.dto.WebSocketMsgDto;
import com.ruoyi.websocket.service.WebSocketService;
import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

@Service
public class FileServiceImpl extends FileServiceGrpc.FileServiceImplBase {

	private static final Logger log = LoggerFactory.getLogger(FileServiceImpl.class);
	@Value("${server.upload.base_path}")
	String uploadBasePath;
	
	@Autowired
	private TokenApiService tokenApiService;
	
	@Autowired
	private INmyUserWatchSubscribeService myUserWatchSubscribeService;
	
	@Autowired
	private WebSocketService webSocketService;
	
	@Override
	public StreamObserver<FileUploadRequest> uploadFile(StreamObserver<FileUploadResponse> responseObserver) {
		return new StreamObserver<FileUploadRequest>() {

			// 定义成员变量保存上下文信息
			private String filename;
			private String token;
			private Long userId;
			private Long deviceId;
			private String changeBasePath;
			private FileOutputStream fos;
			private long totalSize = 0;
			String serverPath = "";
			@Override
			public void onNext(FileUploadRequest request) {
				try {
					if (request.hasFilename()) {
						
						// 保存首个请求块中的元数据
						this.filename = request.getFilename();
						this.token = request.getToken();
						this.changeBasePath = request.getBasePath();
						// 解析用户信息
						ClientUserOnline loginUser = tokenApiService.getLoginUserByToken(token);
						if (loginUser == null) {
							throw new RuntimeException("无效Token: " + token);
						}
						this.userId = loginUser.getUserId();
						this.deviceId = loginUser.getDeviceId();
						serverPath = loginUser.getServerPath();
						// 创建文件对象
						java.io.File ioFile = new java.io.File(getServerFilePath()+filename);
						// 创建文件所在的目录
						ioFile.getParentFile().mkdirs();
						// 创建文件，如果文件存在则覆盖
						ioFile.createNewFile();
						fos = new FileOutputStream(ioFile);
						
					} else {
						byte[] chunk = request.getChunk().toByteArray();
						totalSize += chunk.length;
						fos.write(chunk);
					}	
				}catch (Exception e){
					e.printStackTrace();
					cleanupResources();
					// 返回 INVALID_ARGUMENT 错误
					responseObserver.onError(Status.INVALID_ARGUMENT
							.withDescription("Upload failed:" + e.getMessage())
							.withCause(e)
							.asRuntimeException());
				}
			}

			@Override
			public void onError(Throwable t) {
				cleanupResources();
				System.err.println("客户端中断上传: " + t.getMessage());
			}
			public String getServerFilePath(){
				return uploadBasePath + serverPath + "/" ;
			}

			@Override
			public void onCompleted() {
				try {
					fos.close();
					responseObserver.onNext(FileUploadResponse.newBuilder()
							.setStatus(Status.OK.toString())
							.setSize(totalSize)
							.build());
				
					
					//发消息让该同步的客户端进行同步的消息
					List<WebSocketMsgDto> webSocketSendMsg = myUserWatchSubscribeService.getWebSocketSendMsg(userId, deviceId, changeBasePath);
					if (webSocketSendMsg.isEmpty()) {
						responseObserver.onCompleted();
						return;
					}
					for (WebSocketMsgDto webSocketMsgDto : webSocketSendMsg) {
						//文件
						webSocketMsgDto.setMsgFileType(MsgFileType.file.getCode());
						//创建
						webSocketMsgDto.setMsgHandlerType(MsgHandlerType.create.getCode());
						//存储在服务器的路径
						webSocketMsgDto.setServerFilePath(getServerFilePath());
						webSocketMsgDto.setFileName(filename);
					}
					webSocketService.sendMsg(webSocketSendMsg);
					responseObserver.onCompleted();
					
				} catch (IOException e) {
					e.printStackTrace();
					// 最终错误处理
					responseObserver.onError(Status.INTERNAL
							.withDescription("Finalization failed: " + e.getMessage())
							.asRuntimeException());
				}
			}

			private void cleanupResources() {
				try {
					if (fos != null) {
						fos.close();
					}
				} catch (IOException e) {
					log.warn("文件流关闭异常", e);
				}
			}
			
		};
	}



	@Override
	public void downloadFile(FileDownloadRequest request,
	                         StreamObserver<FileDownloadResponse> responseObserver) {

		FileInputStream fis = null;
		try {
			String fullPath = Paths.get(request.getFilename()).toString();

			java.io.File file = new java.io.File(fullPath);

			// 检查文件是否存在
			if (!file.exists()) {
				throw Status.NOT_FOUND.withDescription("File not found: " + request.getFilename()).asRuntimeException();
			}
			
			fis = new FileInputStream(file);
			byte[] buffer = new byte[64 * 1024]; // 64KB chunks
			int bytesRead;

			//循环输入流，读取fis赋值给buffer，传输给客户端
			while ((bytesRead = fis.read(buffer)) != -1) {
				responseObserver.onNext(FileDownloadResponse.newBuilder()
						.setChunk(ByteString.copyFrom(buffer, 0, bytesRead))
						.build());
			}
			responseObserver.onCompleted();
		} catch (Exception e) {
			// 根据异常类型映射状态码
			Status status = Status.INTERNAL;
			if (e instanceof FileNotFoundException) {
				status = Status.NOT_FOUND.withDescription("File not found");
			}
			responseObserver.onError(status.withCause(e).asRuntimeException());
			log.error(e.getMessage(), e);
		} finally {
			if (fis != null) {
				try {
					fis.close();
				} catch (IOException e) {
					log.error(e.getMessage(), e);
					// 记录日志，但不再抛出新异常
					System.err.println("Failed to close stream: " + e.getMessage());
				}
			}
		}
	}
	
	
}
