package com.ruoyi.grpc.service;

import com.google.protobuf.ByteString;
import com.ruoyi.grpc.file.*;
import com.ruoyi.user.dto.ClientUserOnline;
import com.ruoyi.user.service.TokenApiService;
import com.ruoyi.watch.service.INmyUserWatchSubscribeService;
import com.ruoyi.websocket.dto.WebSocketMsgDto;
import com.ruoyi.websocket.service.WebSocketService;
import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

@Service
public class FileServiceImpl extends FileServiceGrpc.FileServiceImplBase {

	private static final Logger log = LoggerFactory.getLogger(FileServiceImpl.class);
	@Value("${server.upload.base_path}")
	String uploadBasePath;
	
	@Autowired
	private TokenApiService tokenApiService;
	
	@Autowired
	private INmyUserWatchSubscribeService myUserWatchSubscribeService;
	
	@Autowired
	private WebSocketService webSocketService;
	
	@Override
	public StreamObserver<FileUploadRequest> uploadFile(StreamObserver<FileUploadResponse> responseObserver) {
		return new StreamObserver<FileUploadRequest>() {

			// 定义成员变量保存上下文信息
			private String filename;
			private String token;
			private Long userId;
			private Long deviceId;
			private String changeBasePath;
			private FileOutputStream fos;
			private long totalSize = 0;
			
			@Override
			public void onNext(FileUploadRequest request) {
				try {
					if (request.hasFilename()) {
						
						// 保存首个请求块中的元数据
						this.filename = request.getFilename();
						this.token = request.getToken();
						this.changeBasePath = request.getBasePath();
						// 解析用户信息
						ClientUserOnline loginUser = tokenApiService.getLoginUser();
						if (loginUser == null) {
							throw new RuntimeException("无效Token: " + token);
						}
						this.userId = loginUser.getUserId();
						this.deviceId = loginUser.getDeviceId();

						// 创建文件对象
						java.io.File ioFile = new java.io.File(uploadBasePath + filename);
						// 创建文件所在的目录
						ioFile.getParentFile().mkdirs();
						// 创建文件，如果文件存在则覆盖
						ioFile.createNewFile();
						fos = new FileOutputStream(ioFile);
						
					} else {
						byte[] chunk = request.getChunk().toByteArray();
						totalSize += chunk.length;
						fos.write(chunk);
					}	
				}catch (Exception e){
					e.printStackTrace();
					cleanupResources();
					// 返回 INVALID_ARGUMENT 错误
					responseObserver.onError(Status.INVALID_ARGUMENT
							.withDescription("Upload failed:" + e.getMessage())
							.withCause(e)
							.asRuntimeException());
				}
			}

			@Override
			public void onError(Throwable t) {
				cleanupResources();
				System.err.println("客户端中断上传: " + t.getMessage());
			}

			@Override
			public void onCompleted() {
				try {
					fos.close();
					responseObserver.onNext(FileUploadResponse.newBuilder()
							.setStatus(Status.OK.toString())
							.setSize(totalSize)
							.build());
					responseObserver.onCompleted();
					
					//发消息让该同步的客户端进行同步的消息
					List<WebSocketMsgDto> webSocketSendMsg = myUserWatchSubscribeService.getWebSocketSendMsg(userId, deviceId, changeBasePath);

					//TODO
					webSocketService.sendMsg(new ArrayList<>(),"add");
					
				} catch (IOException e) {
					e.printStackTrace();
					// 最终错误处理
					responseObserver.onError(Status.INTERNAL
							.withDescription("Finalization failed: " + e.getMessage())
							.asRuntimeException());
				}
			}

			private void cleanupResources() {
				try {
					if (fos != null) {
						fos.close();
					}
				} catch (IOException e) {
					log.warn("文件流关闭异常", e);
				}
			}
			
		};
	}



	@Override
	public void downloadFile(FileDownloadRequest request,
	                         StreamObserver<FileDownloadResponse> responseObserver) {
		try {

			java.io.File file = new java.io.File(uploadBasePath + request.getFilename());
			FileInputStream fis = new FileInputStream(file);
			byte[] buffer = new byte[64 * 1024]; // 64KB chunks
			int bytesRead;

			//循环输入流，读取fis赋值给buffer，传输给客户端
			while ((bytesRead = fis.read(buffer)) != -1) {
				responseObserver.onNext(FileDownloadResponse.newBuilder()
						.setChunk(ByteString.copyFrom(buffer, 0, bytesRead))
						.build());
			}
			fis.close();
			responseObserver.onCompleted();
		} catch (IOException e) {
			responseObserver.onError(e);
		}
	}
	
	
}
