package com.ruoyi.client.service;

import com.google.protobuf.ByteString;
import com.ruoyi.grpc.file.*;
import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

@Service
public class FileServiceImpl extends FileServiceGrpc.FileServiceImplBase {
	
	@Value("${server.upload.base_path}")
	String uploadBasePath;
	
	@Override
	public StreamObserver<FileUploadRequest> uploadFile(StreamObserver<FileUploadResponse> responseObserver) {
		return new StreamObserver<FileUploadRequest>() {
			private String filename;
			private FileOutputStream fos;
			private long totalSize = 0;

			@Override
			public void onNext(FileUploadRequest request) {
				try {
					if (request.hasFilename()) {
						filename = request.getFilename();
						// 创建文件对象
						java.io.File ioFile = new java.io.File(uploadBasePath + filename);
						// 创建文件所在的目录
						ioFile.getParentFile().mkdirs();
						// 创建文件，如果文件存在则覆盖
						ioFile.createNewFile();
						fos = new FileOutputStream(ioFile);
						
					} else {
						byte[] chunk = request.getChunk().toByteArray();
						totalSize += chunk.length;
						fos.write(chunk);
					}	
				}catch (Exception e){
					e.printStackTrace();
					// 返回 INVALID_ARGUMENT 错误
					responseObserver.onError(Status.INVALID_ARGUMENT
							.withDescription("Upload failed:" + e.getMessage())
							.withCause(e)
							.asRuntimeException());
				}
			}

			@Override
			public void onError(Throwable t) {
				System.err.println("客户端中断上传: " + t.getMessage());
			}

			@Override
			public void onCompleted() {
				try {
					fos.close();
					responseObserver.onNext(FileUploadResponse.newBuilder()
							.setStatus(Status.OK.toString())
							.setSize(totalSize)
							.build());
					responseObserver.onCompleted();
				} catch (IOException e) {
					e.printStackTrace();
					// 最终错误处理
					responseObserver.onError(Status.INTERNAL
							.withDescription("Finalization failed: " + e.getMessage())
							.asRuntimeException());
				}
			}
		};
	}



	@Override
	public void downloadFile(FileDownloadRequest request,
	                         StreamObserver<FileDownloadResponse> responseObserver) {
		try {

			java.io.File file = new java.io.File(uploadBasePath + request.getFilename());
			FileInputStream fis = new FileInputStream(file);
			byte[] buffer = new byte[64 * 1024]; // 64KB chunks
			int bytesRead;

			//循环输入流，读取fis赋值给buffer，传输给客户端
			while ((bytesRead = fis.read(buffer)) != -1) {
				responseObserver.onNext(FileDownloadResponse.newBuilder()
						.setChunk(ByteString.copyFrom(buffer, 0, bytesRead))
						.build());
			}
			fis.close();
			responseObserver.onCompleted();
		} catch (IOException e) {
			responseObserver.onError(e);
		}
	}
	
	
}
