package com.ruoyi.websocket.handler;

import com.ruoyi.common.enums.DeviceOnLineStatus;
import com.ruoyi.device.service.INmyDeviceService;
import com.ruoyi.user.dto.ClientUserOnline;
import com.ruoyi.user.service.WebSocketTokenApiService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.CloseStatus;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import javax.annotation.PreDestroy;
import java.io.IOException;
import java.util.UUID;
import java.util.concurrent.*;

@Component
public class SyncNotificationHandler extends TextWebSocketHandler {
	private static final Logger log = LoggerFactory.getLogger(SyncNotificationHandler.class);
	private static final String PING_PONG_MAP_KEY = "pingPongHashMap";
	private static final long PING_TIMEOUT = 3000; // 3秒超时
	private static final long HEARTBEAT_INTERVAL = 5000; // 5秒心跳间隔

	@Autowired
	private INmyDeviceService myDeviceService;

	@Autowired
	private WebSocketTokenApiService tokenApiService;

	// 线程安全的会话存储（Key: 客户端ID，Value: WebSocketSession）
	private static final ConcurrentHashMap<String, WebSocketSession> sessions = new ConcurrentHashMap<>();

	// 线程池 - 使用ScheduledThreadPoolExecutor以便更好地控制线程池
	private final ScheduledThreadPoolExecutor executorService = new ScheduledThreadPoolExecutor(
			1,
			r -> {
				Thread thread = new Thread(r);
				thread.setName("ws-heartbeat-thread");
				thread.setDaemon(true);
				return thread;
			}
	);

	// 每一个sessionKey连接，维护一个调度对象
	private static final ConcurrentHashMap<String, ScheduledFuture<?>> futureMap = new ConcurrentHashMap<>();

	public String getSessionKey(ClientUserOnline clientUserOnline) {
		return clientUserOnline.getUserId() + "_" + clientUserOnline.getDeviceId();
	}

	@Override
	public void afterConnectionEstablished(WebSocketSession session) {
		log.info("WebSocket连接建立: {}", session.getId());

		try {
			String token = tokenApiService.getToken(session);
			ClientUserOnline loginUser = tokenApiService.getLoginUser(session);

			if (loginUser == null) {
				log.error("token:{},创建webSocket连接失败,没有找到登陆用户", token);
				session.close(CloseStatus.POLICY_VIOLATION.withReason("未登录"));
				return;
			}

			// 初始化ping/pong映射
			ConcurrentHashMap<String, Long> pingPongMap = new ConcurrentHashMap<>();
			session.getAttributes().put(PING_PONG_MAP_KEY, pingPongMap);

			String sessionKey = getSessionKey(loginUser);

			// 移除旧的会话（如果存在）
			WebSocketSession oldSession = sessions.put(sessionKey, session);
			if (oldSession != null && oldSession.isOpen()) {
				try {
					oldSession.close(CloseStatus.NORMAL.withReason("新连接建立"));
				} catch (IOException e) {
					log.warn("关闭旧会话失败: {}", sessionKey, e);
				}
			}

			// 发送连接确认
			sendMessage(session, "CONNECTED");

			// 启动心跳任务
			startHeartbeat(sessionKey, session);

			// 设置设备为在线状态
			myDeviceService.onLine(loginUser.getDeviceId(), DeviceOnLineStatus.on_line.getCode());

		} catch (Exception e) {
			log.error("处理WebSocket连接建立时出错", e);
			try {
				session.close(CloseStatus.SERVER_ERROR);
			} catch (IOException ex) {
				log.warn("关闭WebSocket会话失败", ex);
			}
		}
	}

	private void startHeartbeat(String sessionKey, WebSocketSession session) {
		// 清理旧的心跳任务（如果存在）
		stopHeartbeat(sessionKey);

		// 创建心跳任务
		ScheduledFuture<?> future = executorService.scheduleAtFixedRate(() -> {
			try {
				if (!session.isOpen()) {
					log.debug("会话已关闭，停止心跳: {}", sessionKey);
					stopHeartbeat(sessionKey);
					return;
				}

				String pingId = UUID.randomUUID().toString();
				String pingMessage = "ping:" + pingId;

				ConcurrentHashMap<String, Long> pingPongMap =
						(ConcurrentHashMap<String, Long>) session.getAttributes().get(PING_PONG_MAP_KEY);

				if (pingPongMap == null) {
					log.warn("pingPongMap不存在，停止心跳: {}", sessionKey);
					stopHeartbeat(sessionKey);
					return;
				}

				// 记录发送时间
				pingPongMap.put(pingId, System.currentTimeMillis());

				// 发送ping消息
				log.debug("发送心跳ping: {}", pingId);
				sendMessage(session, pingMessage);

				// 安排超时检查
				executorService.schedule(() -> {
					if (pingPongMap.containsKey(pingId)) {
						log.warn("心跳超时未收到响应: {}", pingId);
						pingPongMap.remove(pingId);
						closeSession(session, CloseStatus.SESSION_NOT_RELIABLE);
					} else {
						log.debug("心跳响应正常: {}", pingId);
					}
				}, PING_TIMEOUT, TimeUnit.MILLISECONDS);

			} catch (Exception e) {
				log.error("执行心跳任务时出错: {}", sessionKey, e);
				closeSession(session, CloseStatus.SERVER_ERROR);
			}
		}, 5, HEARTBEAT_INTERVAL, TimeUnit.MILLISECONDS);

		// 保存任务引用
		futureMap.put(sessionKey, future);
	}

	private void stopHeartbeat(String sessionKey) {
		ScheduledFuture<?> future = futureMap.remove(sessionKey);
		if (future != null) {
			future.cancel(false);
		}
	}

	private void sendMessage(WebSocketSession session, String message) {
		try {
			session.sendMessage(new TextMessage(message));
		} catch (IOException e) {
			log.error("发送WebSocket消息失败: {}", message, e);
			closeSession(session, CloseStatus.SERVER_ERROR);
		}
	}

	private void closeSession(WebSocketSession session, CloseStatus status) {
		try {
			if (session.isOpen()) {
				session.close(status);
			}
		} catch (IOException e) {
			log.warn("关闭WebSocket会话失败", e);
		}
	}

	@PreDestroy
	public void destroy() {
		log.info("关闭WebSocket处理器，清理资源");

		// 停止所有心跳任务
		futureMap.forEach((sessionKey, future) -> {
			if (future != null) {
				future.cancel(true);
			}
		});
		futureMap.clear();

		// 关闭所有会话
		sessions.forEach((sessionKey, session) -> {
			closeSession(session, CloseStatus.SERVICE_RESTARTED);
		});
		sessions.clear();

		// 关闭线程池
		executorService.shutdown();
		try {
			if (!executorService.awaitTermination(30, TimeUnit.SECONDS)) {
				executorService.shutdownNow();
				if (!executorService.awaitTermination(30, TimeUnit.SECONDS)) {
					log.error("线程池未能正常关闭");
				}
			}
		} catch (InterruptedException e) {
			executorService.shutdownNow();
			Thread.currentThread().interrupt();
		}
	}

	@Override
	protected void handleTextMessage(WebSocketSession session, TextMessage message) {
		String payload = message.getPayload();
		log.debug("收到WebSocket消息: {}", payload);

		try {
			if (payload.startsWith("pong:")) {
				handlePongMessage(session, payload);
			} else {
				// 处理其他类型的消息
				handleCustomMessage(session, payload);
			}
		} catch (Exception e) {
			log.error("处理WebSocket消息时出错", e);
			closeSession(session, CloseStatus.SERVER_ERROR);
		}
	}

	private void handlePongMessage(WebSocketSession session, String payload) {
		String[] parts = payload.split(":");
		if (parts.length == 2) {
			String pingId = parts[1];
			ConcurrentHashMap<String, Long> pingPongMap =
					(ConcurrentHashMap<String, Long>) session.getAttributes().get(PING_PONG_MAP_KEY);

			if (pingPongMap != null && pingPongMap.containsKey(pingId)) {
				long responseTime = System.currentTimeMillis() - pingPongMap.remove(pingId);
				log.debug("收到pong响应: {} (响应时间: {}ms)", pingId, responseTime);
			} else {
				log.warn("收到未知pingId的pong响应: {}", pingId);
			}
		}
	}

	private void handleCustomMessage(WebSocketSession session, String payload) {
		// 处理自定义消息的逻辑
		log.info("收到自定义消息: {}", payload);
		// 可以根据需求添加消息处理逻辑
	}

	@Override
	public void afterConnectionClosed(WebSocketSession session, CloseStatus status) {
		log.info("WebSocket连接关闭: {} (状态: {})", session.getId(), status);

		try {
			ClientUserOnline loginUser = tokenApiService.getLoginUser(session);
			if (loginUser != null) {
				String sessionKey = getSessionKey(loginUser);

				// 停止心跳任务
				stopHeartbeat(sessionKey);

				// 移除会话
				sessions.remove(sessionKey);

				// 设置设备为离线状态
				myDeviceService.onLine(loginUser.getDeviceId(), DeviceOnLineStatus.off_line.getCode());

				log.info("{}:已离线", sessionKey);
			}
		} catch (Exception e) {
			log.error("处理WebSocket连接关闭时出错", e);
		}
	}

	// 核心方法：向指定客户端发送同步通知
	public void notifyClient(String sessionKey, String message) {
		WebSocketSession session = sessions.get(sessionKey);
		if (session != null && session.isOpen()) {
			try {
				log.debug("向客户端发送消息: {} => {}", sessionKey, message);
				session.sendMessage(new TextMessage(message));
			} catch (IOException e) {
				log.error("向客户端发送消息失败: {}", sessionKey, e);
				// 清理无效会话
				sessions.remove(sessionKey);
				stopHeartbeat(sessionKey);
			}
		} else {
			log.warn("无法向客户端发送消息: 会话不存在或已关闭: {}", sessionKey);
		}
	}
}
