package com.ruoyi.user.service;

import com.ruoyi.common.constant.Constants;
import com.ruoyi.common.constant.NmyConstans;
import com.ruoyi.common.utils.ServletUtils;
import com.ruoyi.common.utils.StringUtils;
import com.ruoyi.common.utils.uuid.IdUtils;
import com.ruoyi.user.dto.ClientUserOnline;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.Map;


@Component
public class TokenApiService {

	private static final Logger log = LoggerFactory.getLogger(TokenApiService.class);
	// 令牌秘钥
	@Value("${tokenapi.secret}")
	private String secret;

	// 令牌自定义标识
	@Value("${tokenapi.header}")
	private String header;
	
	@Autowired
	private INmyUserService nmyUserService;


	/**
	 * 从数据声明生成令牌
	 *
	 * @param claims 数据声明
	 * @return 令牌
	 */
	private String createToken(Map<String, Object> claims)
	{
		String token = Jwts.builder()
				.setClaims(claims)
				.signWith(SignatureAlgorithm.HS512, secret).compact();
		return token;
	}


	/**
	 * 创建令牌
	 *
	 * @param clientUserOnline 用户信息
	 * @return 令牌
	 */
	public String createToken(ClientUserOnline clientUserOnline)
	{
		String token = IdUtils.fastUUID();
		Map<String, Object> claims = new HashMap<>();
		clientUserOnline.setToken(token);
		claims.put(NmyConstans.userId, clientUserOnline.getUserId());
		claims.put(NmyConstans.deviceId, clientUserOnline.getDeviceId());
		claims.put(NmyConstans.deviceFlag, clientUserOnline.getDeviceFlag());
		claims.put(NmyConstans.deviceIp, clientUserOnline.getDeviceIp());
		claims.put(NmyConstans.serverPath, clientUserOnline.getServerPath());
		claims.put(NmyConstans.token, clientUserOnline.getToken());
		return createToken(claims);
	}


	/**
	 * 获取请求token
	 *
	 * @return token
	 */
	public String getToken()
	{
		HttpServletRequest request = ServletUtils.getRequest();
		String token = request.getHeader(header);
		if (StringUtils.isNotEmpty(token) && token.startsWith(Constants.TOKEN_PREFIX))
		{
			token = token.replace(Constants.TOKEN_PREFIX, "");
		}
		return token;
	}


	/**
	 * 从令牌中获取数据声明
	 *
	 * @param token 令牌
	 * @return 数据声明
	 */
	public Claims parseToken(String token)
	{
		return Jwts.parser()
				.setSigningKey(secret)
				.parseClaimsJws(token)
				.getBody();
	}

	
	public ClientUserOnline getLoginUser(){
		String token = getToken();
		if (StringUtils.isNotEmpty(token))
		{
			try
			{
				Claims claims = parseToken(token);
				Long deviceId = claims.get(NmyConstans.deviceId, Long.class);
				ClientUserOnline userOnline = nmyUserService.getUserOnline(deviceId);
				return userOnline;
			}
			catch (Exception e)
			{
				log.error(e.getMessage(),e);
			}
		}
		return null;
	}

	public ClientUserOnline getLoginUserByToken(String token){
		if (StringUtils.isNotEmpty(token))
		{
			try
			{
				Claims claims = parseToken(token);
				Long deviceId = claims.get(NmyConstans.deviceId, Long.class);
				ClientUserOnline userOnline = nmyUserService.getUserOnline(deviceId);
				return userOnline;
			}
			catch (Exception e)
			{
				log.error(e.getMessage(),e);
			}
		}
		return null;
	}


	






	
	
	

}
