package com.ruoyi.kafka.consumer;

import com.alibaba.fastjson2.JSONObject;
import com.ruoyi.common.utils.MyFileUtils;
import com.ruoyi.file.dto.FileChangeMsg;
import com.ruoyi.file.enums.MsgFileType;
import com.ruoyi.file.enums.MsgHandlerType;
import com.ruoyi.user.dto.ClientUserOnline;
import com.ruoyi.user.service.TokenApiService;
import com.ruoyi.watch.service.INmyUserWatchSubscribeService;
import com.ruoyi.websocket.dto.WebSocketMsgDto;
import com.ruoyi.websocket.service.WebSocketService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.List;

/**
* @return 
* @Version: v1.0
* @Author: LiuYan
* @Date 2025-4-23 10:02
 * 
 * 客户端创建文件夹服务端接收消息的消费者
**/
@Service
public class FileChangeConsumer {

	@Value("${server.upload.base_path}")
	String uploadBasePath;

	@Autowired
	private WebSocketService webSocketService;
	@Autowired
	private TokenApiService tokenApiService;

	@Autowired
	private INmyUserWatchSubscribeService myUserWatchSubscribeService;


	@KafkaListener(
			topics = "${kafka.FileChangeTopic}",  // 动态注入Topic
			groupId = "${kafka.group-id}" // 动态注入Group ID
	)
	public void consume(String message) throws IOException {
		//获取基础路径
		FileChangeMsg fileChangeMsg = JSONObject.parseObject(message, FileChangeMsg.class);
		MsgHandlerType handlerType = fileChangeMsg.getHandlerType();
		MsgFileType fileType = fileChangeMsg.getFileType();
		String token = fileChangeMsg.getToken();
		
		
		ClientUserOnline loginUserByToken = tokenApiService.getLoginUserByToken(token);

		String serverUuidPath = loginUserByToken.getServerPath();

		if (loginUserByToken == null) {
			throw new RuntimeException("无效Token: " + token);
		}
		//发送给需要同步的client	
		List<WebSocketMsgDto> webSocketSendMsg = myUserWatchSubscribeService.getWebSocketSendMsg(loginUserByToken.getUserId(), loginUserByToken.getDeviceId(), fileChangeMsg.getBasePath());
		
		if(handlerType == MsgHandlerType.create){
			if(fileType == MsgFileType.file){
				//创建文件,不用管逻辑，在上传文件实现了
			} else if (fileType == MsgFileType.folder) {
				//创建文件夹
				MyFileUtils.createDirectoryUsingPath(uploadBasePath+serverUuidPath+"/"+fileChangeMsg.getRealPath());
				//同步给客户端
				if (webSocketSendMsg.isEmpty()) {
					return;
				}
				for (WebSocketMsgDto webSocketMsgDto : webSocketSendMsg) {
					//文件
					webSocketMsgDto.setMsgFileType(MsgFileType.folder.getCode());
					//创建
					webSocketMsgDto.setMsgHandlerType(MsgHandlerType.create.getCode());
					//存储在服务器的路径
					webSocketMsgDto.setServerFilePath(uploadBasePath+serverUuidPath+"/");
					webSocketMsgDto.setFileName(fileChangeMsg.getRealPath());
				}
				webSocketService.sendMsg(webSocketSendMsg);
			}
		} else if (handlerType == MsgHandlerType.delete) {
			if(fileType == MsgFileType.file){
				//删除文件
				MyFileUtils.deleteFile(uploadBasePath+serverUuidPath+"/"+fileChangeMsg.getRealPath());
				//同步给客户端
				if (webSocketSendMsg.isEmpty()) {
					return;
				}
				for (WebSocketMsgDto webSocketMsgDto : webSocketSendMsg) {
					webSocketMsgDto.setMsgFileType(MsgFileType.file.getCode());
					webSocketMsgDto.setMsgHandlerType(MsgHandlerType.delete.getCode());
					webSocketMsgDto.setFileName(fileChangeMsg.getRealPath());
					webSocketMsgDto.setServerFilePath(uploadBasePath+serverUuidPath+"/");
				}
				webSocketService.sendMsg(webSocketSendMsg);
			} else if (fileType == MsgFileType.folder) {
				//删除文件夹
				MyFileUtils.deleteDirectory(uploadBasePath+fileChangeMsg.getRealPath());
				//同步给客户端
				if (webSocketSendMsg.isEmpty()) {
					return;
				}
				for (WebSocketMsgDto webSocketMsgDto : webSocketSendMsg) {
					webSocketMsgDto.setMsgFileType(MsgFileType.folder.getCode());
					webSocketMsgDto.setMsgHandlerType(MsgHandlerType.delete.getCode());
					webSocketMsgDto.setFileName(fileChangeMsg.getRealPath());
					webSocketMsgDto.setServerFilePath(uploadBasePath+serverUuidPath+"/");
				}
				webSocketService.sendMsg(webSocketSendMsg);
			}			
		} else if (handlerType == MsgHandlerType.rename) {
//			if(fileType == MsgFileType.file){
//				//重命名文件
//			} else if (fileType == MsgFileType.folder) {
//				//重命名文件夹
//			}
			//截取一下
			String fileName = fileChangeMsg.getRealPath();
			String[] split = fileName.split("→");
			String oldPath = split[0];
			String newPath = split[1];
			MyFileUtils.renameFile(uploadBasePath+oldPath, uploadBasePath+newPath);
		}
		System.out.println("Received message: " + message);
	}
}
